<?php

namespace App\Jobs;

use App\Mail\Billar\InvoiceRecurringMail;
use App\Mail\Billar\NewInvoiceGenerateMail;
use App\Models\Billar\Invoice\Invoice;
use App\Notifications\App\Invoice\InvoiceGenerateNotification;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use PDF;

class InvoiceRecurringJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected Invoice $invoice;

    public function __construct(Invoice $invoice)
    {
        $this->invoice = $invoice;
    }


    public function handle()
    {
        $invoiceInfo = $this->invoiceInfo();
        Mail::to(optional($this->invoice->client)->email)
            ->send(new InvoiceRecurringMail($this->invoice, $invoiceInfo));

        //mail send to app admin
        $roles = \App\Models\Core\Auth\Role::query()->with('users')->where('is_admin', 1)->get();
        foreach ($roles as $role) {
            foreach ($role->users as $user) {
                Mail::to(optional($user)->email)
                    ->send(new NewInvoiceGenerateMail($this->invoice, $invoiceInfo, $user));
                notify()
                    ->on('new_invoice_generate')
                    ->with($this->invoice)
                    ->send(InvoiceGenerateNotification::class);
            }
        }

    }

    public function invoiceInfo(): InvoiceRecurringJob
    {
        $invoiceInfo = $this->invoice->load(['invoiceDetails' => function ($query) {
            $query->with('product:id,name,description', 'tax:id,name,value');
        }, 'client:id,first_name,last_name,email', 'createdBy:id,first_name,last_name']);

        $pdf = PDF::loadView('invoices.invoice-generate', [
            'invoice' => $invoiceInfo
        ]);

        $output = $pdf->output();
        Storage::put('public/pdf/invoice_recurring_' . $this->invoice->id . '.pdf', $output);
        return $this;
    }
}
