<?php

namespace App\Http\Controllers\Billar\Invoice;

use App\Exports\Billar\InvoiceExport;
use App\Filters\Billar\Invoice\InvoiceFilter;
use App\Http\Controllers\Controller;
use App\Jobs\InvoiceAttachmentJob;
use App\Models\Billar\Invoice\Invoice;
use App\Models\Billar\Invoice\InvoiceDetail;
use App\Services\App\SmsSetting\TwilioService;
use App\Services\Billar\Invoice\InvoiceService;
use Maatwebsite\Excel\Facades\Excel;
use PDF;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Request;

class InvoiceController extends Controller
{
    public function __construct(InvoiceService $service, InvoiceFilter $filter)
    {
        $this->service = $service;
        $this->filter = $filter;
    }


    public function index()
    {
        if (request()->get('invoice_type')) {
            return $this->service
                ->where('recurring', 3)
                ->with('status:id,name,class', 'client:id,first_name,last_name', 'recurringCycle:id,name')
                ->filters($this->filter)
                ->orderBy('id', request()->get('orderBy'))
                ->paginate(request('per_page', 10));
        }
        return $this->service
            ->where('recurring', '!=', 3)
            ->with('status:id,name,class', 'client:id,first_name,last_name', 'recurringCycle:id,name')
            ->filters($this->filter)
            ->orderBy('id', request()->get('orderBy'))
            ->paginate(request('per_page', 10));

    }


    public function store(Request $request)
    {
        $invoice = $this->service
            ->setValidation()
            ->setAttributes($request->all())
            ->save();

        $this->service->when($request->has('products'), fn(InvoiceService $service) => $service->invoiceDetails());

        $invoiceInfo = $this->service->loadInvoiceInfo($invoice);

        $this->service
            ->setAttribute('file_path', 'public/pdf/invoice_' . $invoice->id . '.pdf')
            ->pdfGenerate($invoiceInfo);

        InvoiceAttachmentJob::dispatch($invoiceInfo)->onQueue('high');

        $settings = resolve(TwilioService::class)->getData();

        if (isset($invoice->client->profile->contact) && isset($settings['send_auto_sms'])) {
            $this->service->sendTextMessage($invoice);
        }

        return created_responses('invoices');
    }


    public function show($id)
    {
        return $this->service
            ->with([
                'client.profile', 'createdBy.profile',
                'invoiceDetails' => function ($query) {
                    $query->with('tax:id,name,value', 'product:id,name');
                }
            ])->find($id);
    }

    public function update(Request $request, Invoice $invoice)
    {
        $this->service
            ->setModel($invoice)
            ->setValidation()
            ->setAttributes($request->only(
                'client_id',
                'invoice_number',
                'recurring',
                'date',
                'due_date',
                'status_id',
                'recurring_cycle_id',
                'sub_total',
                'discount_type',
                'discount',
                'discount_amount',
                'total',
                'received_amount',
                'due_amount',
                'notes',
                'terms'
            ))
            ->update();
        $this->service->when($request->has('products'), fn(InvoiceService $service) => $service->invoiceDetails());
        return updated_responses('invoices');
    }


    public function destroy(Invoice $invoice)
    {
        $this->service
            ->setModel($invoice)
            ->delete();
        return deleted_responses('invoices');
    }

    public function detailsDelete()
    {
        $invoices = $this->service->find(request('id'));
        $invoices->update(request()->all());
        $invoice = InvoiceDetail::where('id', request('invoice_detail_id'))->delete();
        return deleted_responses('products', ['invoice' => $invoice]);
    }

    public function invoiceExport()
    {
        return Excel::download(new InvoiceExport(), 'invoices.csv', \Maatwebsite\Excel\Excel::CSV);
    }
}
